<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TaskSubmissionResource\Pages;
use App\Models\TaskSubmission;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\HtmlString;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables;

class TaskSubmissionResource extends Resource
{
    protected static ?string $model = TaskSubmission::class;

    protected static ?string $navigationIcon = 'heroicon-o-document-check';
    protected static ?string $modelLabel = 'ارسال تسک';
    protected static ?string $pluralModelLabel = 'انجام تسک‌ها';
    protected static ?string $navigationGroup = 'مدیریت وظایف';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('user.name')
                    ->label('کاربر')
                    ->disabled(),
                TextInput::make('task.title')
                    ->label('عنوان وظیفه')
                    ->disabled(),
                TextInput::make('status')
                    ->label('وضعیت فعلی')
                    ->disabled(),
                Textarea::make('proof_text')
                    ->label('توضیحات مدرک')
                    ->disabled(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->label('ID')->sortable(),
                TextColumn::make('user.name')->label('کاربر')->searchable(),
                TextColumn::make('task.title')->label('وظیفه')->searchable(),
                TextColumn::make('task.reward')->label('پاداش (سکه)')->sortable(),

                BadgeColumn::make('status')
                    ->label('وضعیت')
                    ->colors([
                        'warning' => 'pending',
                        'success' => 'approved',
                        'danger' => 'rejected',
                    ])
                    ->sortable(),

                TextColumn::make('created_at')->label('زمان ارسال')
                    ->dateTime('Y/m/d H:i')
                    ->sortable()
                    ->timezone('Asia/Tehran'),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label('فیلتر وضعیت')
                    ->options([
                        'pending' => 'در انتظار',
                        'approved' => 'تایید شده',
                        'rejected' => 'رد شده',
                    ])
                    ->default('pending'),

                SelectFilter::make('task_id')
                    ->label('فیلتر وظیفه')
                    ->relationship('task', 'title'),
            ])
            ->actions([
                // 1. مشاهده مدرک (عکس/لینک) - بدون تغییر
                Action::make('view_proof')
                    ->label('مشاهده مدرک')
                    ->icon('heroicon-o-eye')
                    ->modalHeading('مدارک ارسالی')
                    ->modalContent(function (TaskSubmission $record) {

                        $html = '<div class="p-4 text-right">';
                        $html .= '<b>توضیحات ارسالی:</b> ' . ($record->proof_text ?? 'ندارد');

                        if ($record->proof_file_id) {
                            $botToken = env('TELEGRAM_BOT_TOKEN');

                            $response = Http::get("https://api.telegram.org/bot{$botToken}/getFile", [
                                'file_id' => $record->proof_file_id
                            ]);

                            if ($response->successful() && isset($response->json()['result']['file_path'])) {
                                $filePath = $response->json()['result']['file_path'];
                                $fileUrl = "https://api.telegram.org/file/bot{$botToken}/{$filePath}";

                                $html .= '<br><br><b>مدرک ارسالی (تصویر):</b>';
                                $html .= '<img src="' . $fileUrl . '" alt="Task Proof" style="max-width: 100%; height: auto; border-radius: 8px; margin-top: 10px;">';
                            } else {
                                $html .= '<br><b>File ID:</b> <code>' . $record->proof_file_id . '</code>';
                                $html .= '<br><br>خطا: نتوانستیم آدرس تصویر را از تلگرام دریافت کنیم.';
                            }
                        } else {
                            $html .= '<br><b>مدرک:</b> مدرک فایلی ارسال نشده است.';
                        }

                        $html .= '</div>';
                        return new HtmlString($html);
                    }),

                // 2. تایید (Approve)
                Action::make('approve_task')
                    ->label('✅ تایید و واریز سکه')
                    ->color('success')
                    ->action(function (TaskSubmission $record) {
                        // 1. تغییر وضعیت و واریز سکه
                        $reward = $record->task->reward;
                        $record->user->increment('coins', $reward);
                        $record->status = 'approved';
                        $record->save();

                        // 2. ارسال پیام به ربات از طریق Webhook داخلی (با POST)
                        // 💡 اطمینان حاصل کنید که آدرس درست است. 
                        $botWebhookUrl = 'https://tel.shahrekordtimes.ir/bot/telegram_bot.php';

                        $response = Http::post($botWebhookUrl, [
                            'admin_action' => 'approve', 
                            'chat_id' => $record->chat_id,
                            'task_title' => $record->task->title, // POST: بدون urlencode
                            'reward' => $reward, 
                        ]);

                        // 3. نمایش خطا در صورت عدم موفقیت Webhook
                        if (!$response->successful()) {
                            \Filament\Notifications\Notification::make()
                                ->title('❌ خطای ارسال پیام')
                                ->body("ربات تلگرام پاسخ مناسبی نداد. کد خطا: " . $response->status() . " (Webhook failed)")
                                ->danger()
                                ->send();
                        } else {
                             \Filament\Notifications\Notification::make()
                                ->title('وظیفه تایید شد')
                                ->body("سکه به کاربر {$record->user->name} واریز شد و پیام ارسال گردید.")
                                ->success()
                                ->send();
                        }
                    }),

                // 3. رد (Reject)
                Action::make('reject_task')
                    ->color('danger')
                    ->icon('heroicon-o-x-circle')
                    ->modalHeading('رد کردن وظیفه')
                    ->form([
                        Textarea::make('rejection_reason')
                            ->label('علت رد')
                            ->required()
                            ->placeholder('مثال: تصویر ارسالی نامعتبر است.'),
                    ])
                    ->action(function (TaskSubmission $record, array $data) {

                        $reason = $data['rejection_reason'] ?? 'علت رد توسط ادمین مشخص نشد.';

                        $record->status = 'rejected';
                        $record->rejection_reason = $reason; 
                        $record->save();

                        // 2. ارسال پیام به ربات از طریق Webhook داخلی (با POST)
                        $botWebhookUrl = 'https://tel.shahrekordtimes.ir/bot/telegram_bot.php';

                        $response = Http::post($botWebhookUrl, [ // 👈 اصلاح به POST
                            'admin_action' => 'reject',
                            'chat_id' => $record->chat_id,
                            'task_title' => $record->task->title, // POST: بدون urlencode
                            'reason' => $reason, // POST: بدون urlencode
                        ]);

                        // 3. نمایش خطا در صورت عدم موفقیت Webhook
                         if (!$response->successful()) {
                            \Filament\Notifications\Notification::make()
                                ->title('❌ خطای ارسال پیام')
                                ->body("ربات تلگرام پاسخ مناسبی نداد. کد خطا: " . $response->status() . " (Webhook failed)")
                                ->danger()
                                ->send();
                        } else {
                            \Filament\Notifications\Notification::make()
                                ->title('وظیفه رد شد')
                                ->body("پیام رد برای کاربر {$record->user->name} ارسال گردید.")
                                ->danger()
                                ->send();
                        }
                    }),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTaskSubmissions::route('/'),
            'create' => Pages\CreateTaskSubmission::route('/create'),
            'edit' => Pages\EditTaskSubmission::route('/{record}/edit'),
        ];
    }
}