<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Filament\Resources\UserResource\RelationManagers;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\TextInput; // 👈 افزودن
use Filament\Forms\Components\Toggle; // 👈 افزودن
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn; // 👈 افزودن
use Filament\Tables\Columns\IconColumn; // 👈 افزودن
use Filament\Tables\Filters\SelectFilter; // 👈 افزودن
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Hash; // 👈 افزودن برای هش کردن پسورد

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users'; // آیکون مناسب‌تر
    protected static ?string $modelLabel = 'کاربر';
    protected static ?string $pluralModelLabel = 'کاربران ربات';
    protected static ?string $navigationGroup = 'مدیریت کاربران';
    
    // --- متد Form (برای ساخت و ویرایش کاربر) ---

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات پایه کاربر')
                    ->columns(3)
                    ->schema([
                        // ستون 'name' پس از تغییر نام از 'full_name'
                        TextInput::make('name')
                            ->label('نام کامل') 
                            ->required()
                            ->maxLength(255),
                        
                        TextInput::make('chat_id')
                            ->label('Chat ID تلگرام')
                            ->numeric()
                            ->required()
                            ->unique(ignoreRecord: true),
                        
                        TextInput::make('username')
                            ->label('یوزرنیم تلگرام')
                            ->nullable()
                            ->maxLength(255),
                    ]),
                
                Forms\Components\Section::make('احراز هویت و دسترسی')
                    ->columns(2)
                    ->schema([
                        TextInput::make('email')
                            ->label('ایمیل (ورود به پنل)')
                            ->email()
                            ->unique(ignoreRecord: true)
                            ->required(fn (string $operation): bool => $operation === 'create'), // ایمیل در زمان ساخت اجباری است
                            
                        TextInput::make('password')
                            ->label('رمز عبور')
                            ->password()
                            ->dehydrateStateUsing(fn (string $state): string => Hash::make($state)) // هش کردن پسورد
                            ->dehydrated(fn (?string $state): bool => filled($state))
                            ->required(fn (string $operation): bool => $operation === 'create'), // پسورد در زمان ساخت اجباری است
                        
                        Toggle::make('is_admin')
                            ->label('دسترسی ادمین')
                            ->helperText('دسترسی به پنل ادمین را فعال/غیرفعال می‌کند.'),
                    ]),

                Forms\Components\Section::make('اطلاعات ربات و سکه')
                    ->columns(3)
                    ->schema([
                        TextInput::make('coins')
                            ->label('موجودی سکه')
                            ->numeric()
                            ->required()
                            ->default(0),
                        
                        TextInput::make('referral_count')
                            ->label('تعداد زیرمجموعه')
                            ->numeric()
                            ->disabled()
                            ->default(0),
                        
                        TextInput::make('job')
                            ->label('شغل'),
                        
                        TextInput::make('phone')
                            ->label('شماره تماس'),
                        
                        // این فیلدها معمولاً در پنل ویرایش نمی‌شوند
                        TextInput::make('referral_code')
                            ->label('کد ارجاع')
                            ->disabled(),
                    ]),
            ]);
    }

    // --- متد Table (نمایش لیست کاربران) ---

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->label('ID')->sortable(),
                TextColumn::make('name')->label('نام کامل')->searchable(), // 👈 استفاده از 'name'
                TextColumn::make('chat_id')->label('Chat ID')->searchable(),
                TextColumn::make('username')->label('یوزرنیم')->searchable()->toggleable(),
                TextColumn::make('coins')->label('سکه')->sortable(),
                TextColumn::make('referral_count')->label('زیرمجموعه')->sortable(),
                
                IconColumn::make('is_admin')->label('ادمین')
                    ->boolean()
                    ->toggleable(),
                
                TextColumn::make('created_at')->label('تاریخ ثبت‌نام')
                    ->dateTime('Y/m/d H:i')
                    ->sortable()
                    ->timezone('Asia/Tehran')
                    ->toggleable(isToggledHiddenByDefault: true),
                
                TextColumn::make('email')->label('ایمیل ورود')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('is_admin')
                    ->label('وضعیت دسترسی')
                    ->options([
                        true => 'مدیران',
                        false => 'کاربران عادی',
                    ]),
                
                SelectFilter::make('registration_step')
                    ->label('مرحله ثبت‌نام')
                    ->options([
                        'name' => 'انتخاب نام',
                        'phone' => 'ثبت تلفن',
                        'completed' => 'تکمیل شده',
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc'); 
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
    
    // تنظیمات فارسی سازی
    public static function getNavigationLabel(): string
    {
        return 'کاربران';
    }
}